//
// Test 94 Scenario :
// A data provider has 20 data offerings for sale. These are added (minted) with randomly generated prices.
// A customer gets 1000  (x10^-18) FDE digital euros by having governance mint the digital euro tokens to customer's account.
// The customer makes a single approval letting Bourse contract to spend up to 200 of the 1000 (x10^-18) FDE tokens.
// The customer makes multiple (10) purchases  data asset access rights. After the purchases, balance and allowance
// of the customer is checked to see they are OK as expected.
// Note: default no. of decimals in FDE ERC20 token is assumed which is 18.
import { describe } from 'mocha';
import { expect } from 'chai';
import {
  FameContracts,
  deployContractsForTests,
  getSigners,
} from './helpers/deploy-helper';
import {
  initializeContracts,
  initializeTradingManagmentContract,
} from './helpers/contracts-initialization-helper';
import { ethers } from 'ethers';
import { Signers } from './types';
import { Governance, OfferingToken, PaymentToken } from '../../types';
import {
  Bourse,
  DataAccessPayAsYouGo,
  TradingManagementStorageFacet,
  TradingManagementExecutorFacet,
  DiamondLoupeFacet,
} from '../../types';
import { deployDiamond } from '../../scripts/deployTradingManagement';

const numDataOfferings = 20; // number of data asset offerings
const numPurchases = 10; // number of data asset right purchases
// (following definitions assume default no. of decimals in ERC20 which is 18)
const customerOwnedFDEAmount = 1000; // 1000E-18 tokens
const approvedAmount = 200; // 200E-18 tokens
const priceUpperBound = 20; // 20E-18 tokens ,  uppper bound on data asset price
let govadmin;
let customer;
let dataprovider;
let beforeCustomerBalance;
let paymentContract: PaymentToken;
let dataAccPAYGContract: DataAccessPayAsYouGo;
let bourseContract: Bourse;
let governanceContract: Governance;
let offeringTokenContract: OfferingToken;
let contracts: FameContracts;
let signers: Signers;
const oidsPurchased: Array<string> = [];
let totalPurchasedAssetPrices = 0;
const oid2PriceMap = new Map<string, number>();
let tradingManagementStorageFacet: TradingManagementStorageFacet;
let tradingManagementExecutorFacet: TradingManagementExecutorFacet;
let diamondLoupeFacet: DiamondLoupeFacet;
//TODO change this to proper type. There are erros in tests
let afterMintCustomerBalance: any;

describe('94 - Data Access PAYG - costumer performs only single payment token transfer approval operation and subsequent multiple purchases of data access PAYG rights from a data provider', function () {
  before(async function () {
    contracts = await deployContractsForTests({ shouldResetNetwork: false });
    signers = await getSigners();
    await initializeContracts({ contracts, signers });

    const tradingManagementAddress = await deployDiamond({
      saveInfo: false,
      showInfo: false,
    });
    await initializeTradingManagmentContract({
      contracts,
      signers,
      tradingManagementAddress,
    }).then((res) => {
      tradingManagementStorageFacet = res.tradingManagementStorageFacet;
      tradingManagementExecutorFacet = res.tradingManagementExecutorFacet;
      diamondLoupeFacet = res.diamondLoupeFacet;
    });

    dataAccPAYGContract = contracts.dataAccessPayAsYouGoContract;
    bourseContract = contracts.bourseContract;
    paymentContract = contracts.paymentTokenContract;
    governanceContract = contracts.governanceContract;
    offeringTokenContract = contracts.offeringTokenContract;
    signers = await getSigners();
    govadmin = signers.admin;
    customer = signers.signer1;
    dataprovider = signers.signer2;
    beforeCustomerBalance = await paymentContract.balanceOf(customer.address);
  });

  it('Governance contract owned by admin should mint the 1000 (x10^-18) FDE payment tokens to customer', async function () {
    await governanceContract
      .connect(govadmin)
      .mintCoin(customer.address, 'FDE', customerOwnedFDEAmount);
    afterMintCustomerBalance = await paymentContract.balanceOf(
      customer.address,
    );
    expect(afterMintCustomerBalance).to.equal(
      beforeCustomerBalance + customerOwnedFDEAmount,
    );
  });

  it('Customer gives single approval to Bourse Contract to spend 200 (x10^-18) FDE tokens', async function () {
    const result = await paymentContract
      .connect(customer)
      .approve(bourseContract.address, approvedAmount);
    expect(result).to.be.ok;
  });

  it('Multiple (20) data assets from a provider are added (minted) with randomly generated prices', async function () {
    // create multple offering contracts.
    // In FTS tm/src/contracts/OfferingToken.sol code, addAsset is onlyOwner, therefore
    // it is assumed that this function is executed on the server by the admin.
    let price: number;
    let tokenuri: string;

    for (let i = 1; i <= numDataOfferings; i++) {
      const assetid = 'aabc' + i.toString(); // generate oid as aabc1, aabc2, aabc3 etc.
      const oid = 'oabc' + i.toString(); // generate oid as oabc1, oabc2, oabc3 etc.
      tokenuri = 'https://example.com/' + oid;
      price = Math.floor(Math.random() * priceUpperBound + 1);
      oid2PriceMap.set(oid, price);
      const result = await offeringTokenContract
        .connect(govadmin)
        .addAsset(
          assetid,
          oid,
          tokenuri,
          dataprovider.address,
          price,
          1234567890,
          '10',
          '100MB',
          'target1',
          'sl1',
          '0x00',
        );
      await result.wait();
      expect(result).to.be.ok;
    }
  });

  it('Added multiple (20) data asset offerings exist', async function () {
    for (let i = 1; i <= numDataOfferings; i++) {
      const oid = 'oabc' + i.toString(); // generate oid as oabc1, oabc2, oabc3 etc.
      const result = await offeringTokenContract.tokenExists(oid);
      expect(result).to.be.true;
    }
  });

  it('Before multiple (10)  purchases, customer should have 0 token balance of data access PAYG rights', async function () {
    for (let i = 1; i <= numDataOfferings; i++) {
      const oid = 'oabc' + i.toString(); // generate oid as oabc1, oabc2, oabc3 etc.
      const assetid = await offeringTokenContract.getIDHash(oid);
      const dataAccessCustomerBalance = await dataAccPAYGContract.balanceOf(
        customer.address,
        assetid,
      );
      expect(dataAccessCustomerBalance).to.equal(0);
    }
  });

  it('Customer makes multiple (10) purchases of data access PAYG rights', async function () {
    const randomnoGeneratedMap = new Map<number, boolean>();
    let i = 1;
    while (i <= numPurchases) {
      const k = Math.floor(Math.random() * numDataOfferings + 1);
      const oid = 'oabc' + k.toString(); // generate oid as oabc1, oabc2, oabc3 etc.
      if (randomnoGeneratedMap.has(k)) continue;
      randomnoGeneratedMap.set(k, true);
      oidsPurchased.push(oid);
      totalPurchasedAssetPrices =
        totalPurchasedAssetPrices + oid2PriceMap.get(oid);
      const data = { purchasedAssetId: oid };
      const jsonData: string = JSON.stringify(data);
      const dataAsBytes: string = ethers.utils.hexlify(
        ethers.utils.toUtf8Bytes(jsonData),
      );
      const result = await tradingManagementExecutorFacet
        .connect(customer)
        .purchaseAccessRightPAYG(oid, dataAsBytes, 1234567890);
      expect(result).to.be.ok;
      i++;
    }
  });

  it('After multiple (10)  purchases, customer should have 1 token balance of each data access PAYG right purchased', async function () {
    for (let i = 1; i <= oidsPurchased.length; i++) {
      const assetid = await offeringTokenContract.getIDHash(
        oidsPurchased[i - 1],
      );
      const dataAccessCustomerBalance = await dataAccPAYGContract.balanceOf(
        customer.address,
        assetid,
      );
      expect(dataAccessCustomerBalance).to.equal(1);
    }
  });

  it('Customer FDE token balance should not change, withdrawal of the fund is performed by provider when user accesses the data', async function () {
    const afterPurchaseBalance = await paymentContract.balanceOf(
      customer.address,
    );
    expect(afterPurchaseBalance).to.equal(afterMintCustomerBalance);
  });
});
